from django.shortcuts import render, redirect, get_object_or_404
from django.views import View
from django.contrib.auth.mixins import LoginRequiredMixin
from .models import User
from django.contrib import messages
class DashboardView(LoginRequiredMixin, View):
    def get(self, request):
        user = request.user
        context = {
            'user': user,
            'is_admin': user.role == 'ADMIN',
            'is_staff': user.role == 'STAFF',
            'is_customer': user.role == 'CUSTOMER',
            'is_rider': user.role == 'RIDER',
        }
        return render(request, 'pages/dashboard.html', context)

class UserManagementView(LoginRequiredMixin, View):
    def get(self, request):
        users = User.objects.all()
        return render(request, 'pages/user-management/user_management.html', {'users': users})

    def post(self, request):
        # Handle create or update
        user_id = request.POST.get('user_id')
        if user_id:
            user = get_object_or_404(User, pk=user_id)
            user.first_name = request.POST.get('first_name')
            user.last_name = request.POST.get('last_name')
            user.email = request.POST.get('email')
            user.role = request.POST.get('role')
            user.banned = True if request.POST.get('banned') else False
            user.save()
            messages.success(request, 'User updated successfully!')
        else:
            banned = True if request.POST.get('banned') else False
            # Use create_user to ensure password hashing
            username = request.POST.get('username')
            password = request.POST.get('password')
            user = User.objects.create_user(
                username=username,
                email=request.POST.get('email'),
                password=password or None,
            )
            user.first_name = request.POST.get('first_name')
            user.last_name = request.POST.get('last_name')
            user.role = request.POST.get('role')
            user.banned = banned
            user.save()
            # If this user is a rider, ensure Rider profile exists and redirect to its edit page
            if user.role == 'RIDER':
                from riders.models import Rider
                rider, _ = Rider.objects.get_or_create(user=user, defaults={'full_name': user.get_full_name() or user.username, 'email': user.email})
                return redirect('rider-edit', rider.id)
            messages.success(request, 'User created successfully!')
        return redirect('user-management')

    def delete(self, request, *args, **kwargs):
        user_id = request.POST.get('user_id')
        user = get_object_or_404(User, pk=user_id)
        user.delete()
        messages.success(request, 'User deleted successfully!')
        return redirect('user-management')



class UserDetailViewWeb(LoginRequiredMixin, View):
    def get(self, request, pk):
        user = get_object_or_404(User, pk=pk)
        return render(request, 'pages/user-management/user_detail.html', {'user': user})

class UserEditView(LoginRequiredMixin, View):
    def get(self, request, pk):
        user = get_object_or_404(User, pk=pk)
        return render(request, 'pages/user-management/user_edit.html', {'user': user})

    def post(self, request, pk):
        user = get_object_or_404(User, pk=pk)
        user.first_name = request.POST.get('first_name')
        user.last_name = request.POST.get('last_name')
        user.email = request.POST.get('email')
        user.role = request.POST.get('role')
        user.banned = True if request.POST.get('banned') else False
        password = request.POST.get('password')
        if password:
            user.set_password(password)
        user.save()
        # If user now has role RIDER, ensure Rider exists
        if user.role == 'RIDER':
            from riders.models import Rider
            Rider.objects.get_or_create(user=user, defaults={'full_name': user.get_full_name() or user.username, 'email': user.email})
        messages.success(request, 'User updated successfully!')
        return redirect('user-management')

class UserDeleteView(LoginRequiredMixin, View):
    def post(self, request, pk):
        user = get_object_or_404(User, pk=pk)
        user.delete()
        messages.success(request, 'User deleted successfully!')
        return redirect('user-management')


class UserCreateView(LoginRequiredMixin, View):
    def get(self, request):
        return render(request, 'pages/user-management/user_create.html')

    def post(self, request):
        banned = True if request.POST.get('banned') else False
        # Prefer create_user to ensure password hashing
        username = request.POST.get('username')
        password = request.POST.get('password')
        user = User.objects.create_user(
            username=username,
            email=request.POST.get('email'),
            password=password or None,
        )
        user.first_name = request.POST.get('first_name')
        user.last_name = request.POST.get('last_name')
        user.role = request.POST.get('role')
        user.banned = banned
        user.save()
        if user.role == 'RIDER':
            from riders.models import Rider
            rider, _ = Rider.objects.get_or_create(user=user, defaults={'full_name': user.get_full_name() or user.username, 'email': user.email})
            return redirect('rider-edit', rider.id)
        messages.success(request, 'User created successfully!')
        return redirect('user-management')
