from rest_framework import generics, permissions
from .models import Banner
from .serializers import BannerSerializer
from common.response_helper import ResponseHelper


class BannerListAPIView(generics.ListAPIView):
    """Public list of banners. Returns a standardized envelope using ResponseHelper."""
    queryset = Banner.objects.all()
    serializer_class = BannerSerializer
    permission_classes = [permissions.AllowAny]

    def list(self, request, *args, **kwargs):
        qs = self.filter_queryset(self.get_queryset())
        page = self.paginate_queryset(qs)
        if page is not None:
            serializer = self.get_serializer(page, many=True)
            paginated = self.get_paginated_response(serializer.data)
            # paginated is a DRF Response; take its data payload and wrap it
            return ResponseHelper.success(data=paginated.data, message='Banners list')

        serializer = self.get_serializer(qs, many=True)
        return ResponseHelper.success(data=serializer.data, message='Banners list')


class BannerDetailAPIView(generics.RetrieveAPIView):
    """Public banner detail endpoint. Returns a standardized envelope using ResponseHelper."""
    queryset = Banner.objects.all()
    serializer_class = BannerSerializer
    permission_classes = [permissions.AllowAny]

    def retrieve(self, request, *args, **kwargs):
        instance = self.get_object()
        serializer = self.get_serializer(instance)
        return ResponseHelper.success(data=serializer.data, message='Banner detail')
