from rest_framework import generics, permissions
from .models import Category
from .serializers import CategorySerializer
from common.response_helper import ResponseHelper


class CategoryListAPIView(generics.ListAPIView):
    """Public list of categories. Returns standardized envelope using ResponseHelper."""
    queryset = Category.objects.filter(trash=False)
    serializer_class = CategorySerializer
    permission_classes = [permissions.AllowAny]

    def list(self, request, *args, **kwargs):
        qs = self.filter_queryset(self.get_queryset())
        page = self.paginate_queryset(qs)
        if page is not None:
            serializer = self.get_serializer(page, many=True)
            paginated = self.get_paginated_response(serializer.data)
            return ResponseHelper.success(data=paginated.data, message='Categories list')

        serializer = self.get_serializer(qs, many=True)
        return ResponseHelper.success(data=serializer.data, message='Categories list')


class CategoryDetailAPIView(generics.RetrieveAPIView):
    """Public category detail endpoint. Returns standardized envelope using ResponseHelper."""
    queryset = Category.objects.filter(trash=False)
    serializer_class = CategorySerializer
    permission_classes = [permissions.AllowAny]

    def retrieve(self, request, *args, **kwargs):
        instance = self.get_object()
        serializer = self.get_serializer(instance)
        return ResponseHelper.success(data=serializer.data, message='Category detail')
